/*
Version: 1.0, requires Writemonkey 2.5.0.8 or higher

Release date: August 25th 2013

Author: Josip

Description: 
Keep clipboard history and allow text snippets to be accessed later. The plugin will listen for clipboard changes globally, even when Writemonkey is not active (running in background).

Installation:
Unzip the "Clipboard picker" directory into the plugins directory. This plugin is already included with in wm bundle.

Deployment:
This plugin is initialized at startup. Use CTRL+E,E or click top left corner of the wm window to show the clipboard history menu.
*/

monkey.include("json2.incl");
var clipboardItemsArray = [];

/// SETTINGS ///

var shortcutKey = "E";
//monkey.setReplacement("/clip", "{^ee}");
var saveOnExit = false;
var userDataFilePath = monkey.pluginDirectoryPath + "/user.data";
var maxItems = 50;
var maxMenuItemChars = 60;
var startNewLineAt = 70;
var showToolTips = false;
var recordingActive = true;
var locationX = 5;
var locationY = 5;

/// STRINGS ///

var pluginName = "Clipboard picker for wm";
var clearAllS = "Clear all";
var stopRecordingS = "Stop recording";
var allToRepositoryS = "All to repository";
var toolsS = "Actions";
var emptyS = "Empty";
var toRepoS = "To repository";
var pasteS = "Paste";
var copyS = "Copy";
var deleteS = "Delete";
var deleteBelowS = "Delete below";
var startRecordingS = "Start recording";
var stopRecordingS = "Stop recording";
var fromWMSnippetsS = "// From Clipboard picker plugin ";
var dateTimeFormat = "yyyy-MM-dd, HH:mm";
var separatorS = "// Next clipboard item //";
var charactersS = "chr";
var cantWrite = "Can't write to file!";
var cantRead = "Can't read from file!";

/// UI ///

// Context menu
var contextMenu = new System.Windows.Forms.ContextMenuStrip();
contextMenu.ShowImageMargin = false;

// resture / save to JSON file events
monkey.add_appLoadedE(function(object, eventargs) {
	loadFromJSON();
});
monkey.add_appClosingE(function(object, eventargs) {
	saveToJSON();
});

// Actions drop down commands
var actionsDropDown = new System.Windows.Forms.ToolStripDropDown();
actionsDropDown.ShowImageMargin = false;

var actionClearAll = new System.Windows.Forms.ToolStripMenuItem(clearAllS);
var actionStopRecording = new System.Windows.Forms.ToolStripMenuItem(stopRecordingS);
var actionAllToRepository = new System.Windows.Forms.ToolStripMenuItem(allToRepositoryS);

actionsDropDown.Items.Add(actionStopRecording);
actionsDropDown.Items.Add(actionAllToRepository);
actionsDropDown.Items.Add(actionClearAll);


actionClearAll.add_Click(function(object, eventargs) {
	clearAllItems();
});
actionStopRecording.add_Click(function(object, eventargs) {
	toggleStartStop();
});
actionAllToRepository.add_Click(function(object, eventargs) {
	sendAllToRepository();
});
monkey.form.add_OnClipboardChange(function(object, eventargs) {
	addItem();
});

// Entry point event / shortcut fired
monkey.add_extendedShortcutFiredE(function(object, eventargs) {
	if (eventargs.extendedKey == shortcutKey) {
		showContextMenu();
	}
});

// Entry point event / mouseClicked
monkey.form.add_MouseDown(function(object, eventargs) {
	if (eventargs.X < 15 && eventargs.Y < 15) {
		showContextMenu();
	}
});



function loadFromJSON() {
	if (saveOnExit) {
		try {
			var json = System.IO.File.ReadAllText(userDataFilePath, System.Text.Encoding.UTF8);
			if (json == "") json = "[]";
			var jsonObject = JSON.parse(json);

			clipboardItemsArray = jsonObject;
		} catch (e) {
			var result = System.Windows.Forms.MessageBox.Show(userDataFilePath + "\r\n" + cantRead, pluginName, System.Windows.Forms.MessageBoxButtons.OK);
		}
	}
}


function saveToJSON() {
	if (saveOnExit) {
		try {
			var jsonString = JSON.stringify(clipboardItemsArray);
			System.IO.File.WriteAllText(userDataFilePath, jsonString, System.Text.Encoding.UTF8);
		} catch (e) {
			var result = System.Windows.Forms.MessageBox.Show(userDataFilePath + "\r\n" + cantWrite, pluginName, System.Windows.Forms.MessageBoxButtons.OK);
		}
	}
}


function addItem() {

	var clipboardText = "";
	if (System.Windows.Forms.Clipboard.ContainsText()) {
		clipboardText = System.Windows.Forms.Clipboard.GetText();
	}
	if (recordingActive == true && clipboardText.length > 0 && clipboardText != clipboardItemsArray[0].text) {
		var item = new ClipboardItem(clipboardText);
		item._init();
		clipboardItemsArray.unshift(item);

		// Trim if over limit
		if (clipboardItemsArray.length > maxItems) {
			clipboardItemsArray.pop();
		}
	}
}

function showContextMenu(s) {
	if (s == null) s = 0;
	contextMenu.Items.Clear();

	//var filterBox = new System.Windows.Forms.ToolStripTextBox();
	//filterBox.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
	var actionsMenuItem = new System.Windows.Forms.ToolStripMenuItem();
	actionsMenuItem.Font = new System.Drawing.Font(actionsMenuItem.Font.Name, actionsMenuItem.Font.Size, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point);
	actionsMenuItem.Text = toolsS;
	actionsMenuItem.Padding = new System.Windows.Forms.Padding(0, 0, 0, 0);
	actionsMenuItem.DropDown = actionsDropDown;

	var separator = new System.Windows.Forms.ToolStripSeparator();

	contextMenu.Items.Add(actionsMenuItem);
	//contextMenu.Items.Add(filterBox);
	contextMenu.Items.Add(separator);

	if (clipboardItemsArray.length == 0) {
		var emptyMenuItem = new System.Windows.Forms.ToolStripMenuItem(emptyS);
		emptyMenuItem.Enabled = false;
		contextMenu.Items.Add(emptyMenuItem);
	}

	for (var i = 0; i < clipboardItemsArray.length; i++) {
		var menuItem = new System.Windows.Forms.ToolStripMenuItem();

		var agoString = "";
		try {
			agoString = " | " + getAgoString(System.DateTime.Parse(clipboardItemsArray[i].dateCreated));
		} catch (e) {
			agoString = "";
		}
		menuItem.Text = clipboardItemsArray[i].menuText + agoString;
		menuItem.AutoToolTip = showToolTips;
		//menuItem.ToolTipText = clipboardItemsArray[i].text;
		menuItem.BackColor = (i % 2 == 0) ? System.Drawing.Color.FromArgb(250, contextMenu.BackColor) : contextMenu.BackColor; // alternate colors
		menuItem.ForeColor = (clipboardItemsArray[i].pasted) ? monkey.dimmedColor : contextMenu.ForeColor;
		menuItem.Padding = new System.Windows.Forms.Padding(0, 0, 0, 10);
		menuItem.Tag = i;


		var itemDropDown = new System.Windows.Forms.ToolStripDropDown();
		itemDropDown.ShowImageMargin = false;

		var toRepoMenuItem = new System.Windows.Forms.ToolStripMenuItem(toRepoS);
		toRepoMenuItem.Tag = i;
		var pasteMenuItem = new System.Windows.Forms.ToolStripMenuItem(pasteS);
		pasteMenuItem.Tag = i;
		var copyMenuItem = new System.Windows.Forms.ToolStripMenuItem(copyS);
		copyMenuItem.Tag = i;
		var deleteMenuItem = new System.Windows.Forms.ToolStripMenuItem(deleteS);
		deleteMenuItem.Tag = i;
		var deleteBelowMenuItem = new System.Windows.Forms.ToolStripMenuItem(deleteBelowS);
		deleteBelowMenuItem.Tag = i;

		toRepoMenuItem.add_Click(function(object, eventargs) {
			toRepo(object);
		});
		pasteMenuItem.add_Click(function(object, eventargs) {
			pasteItem(object);
		});
		copyMenuItem.add_Click(function(object, eventargs) {
			copyItem(object);
		});
		deleteMenuItem.add_Click(function(object, eventargs) {
			deleteItem(object);
		});
		deleteBelowMenuItem.add_Click(function(object, eventargs) {
			deleteAllBelow(object);
		});

		itemDropDown.Items.Add(toRepoMenuItem);
		itemDropDown.Items.Add(pasteMenuItem);
		itemDropDown.Items.Add(copyMenuItem);
		itemDropDown.Items.Add(deleteMenuItem);
		itemDropDown.Items.Add(deleteBelowMenuItem);

		menuItem.DropDown = itemDropDown;
		contextMenu.Items.Add(menuItem);

		//var separator = new System.Windows.Forms.ToolStripSeparator();
		//contextMenu.Items.Add(separator);

		if (i == s) menuItem.Select();
	}

	contextMenu.Show(new System.Drawing.Point(locationX, //monkey.form.Location.X + monkey.richTextBox.Location.X + monkey.richTextBox.Width + 20
	locationY)); //monkey.form.Location.Y + monkey.richTextBox.Location.Y + 20

	// contextMenu.Show(monkey.richTextBox.PointToScreen(monkey.richTextBox.GetPositionFromCharIndex(monkey.selectionStart + monkey.selectionLength)));
}


function toggleStartStop() {
	if (recordingActive) {
		recordingActive = false;
		actionStopRecording.Text = startRecordingS;
	} else {
		recordingActive = true;
		actionStopRecording.Text = stopRecordingS;
	}
}

function toRepo(o) {
	var textToRepo = "\n\n" + fromWMSnippetsS + monkey.dateTimeNow.ToString(dateTimeFormat) + " //\n";
	monkey.repositoryText += textToRepo + clipboardItemsArray[o.Tag].text + "\n";
	clipboardItemsArray[o.Tag].pasted = true;
	showContextMenu(o.Tag);
}

function pasteItem(o) {

	monkey.selectedText = clipboardItemsArray[o.Tag].text + "\n";
	monkey.scrollToCaret();
	clipboardItemsArray[o.Tag].pasted = true;
	showContextMenu(o.Tag);

}

function copyItem(o) {
	var state = recordingActive;
	recordingActive = false;
	System.Windows.Forms.Clipboard.SetText(clipboardItemsArray[o.Tag].text);
	recordingActive = state;
	clipboardItemsArray[o.Tag].pasted = true;
	//showContextMenu(o.Tag);
}

function deleteItem(o) {
	clipboardItemsArray.splice(o.Tag, 1);
	showContextMenu(o.Tag - 1);
}



function deleteAllBelow(o) {
	clipboardItemsArray.length = o.Tag + 1;
	showContextMenu(o.Tag);
}


function sendAllToRepository() {
	var textToRepo = "\n" + fromWMSnippetsS + monkey.dateTimeNow.ToString(dateTimeFormat) + " //\n";
	monkey.repositoryText += textToRepo + getStringForExport();
	//showContextMenu(0);
}



function clearAllItems() {
	clipboardItemsArray = [];
	showContextMenu(0);
}


function getStringForExport() {
	var textToRepo = "";
	for (var i = 0; i < clipboardItemsArray.length; i++) {
		textToRepo += "\n" + separatorS + "\n" + clipboardItemsArray[i].text;
	}
	return textToRepo;
}


// Util functions

function trim(myString) {
	return myString.replace(/^\s+/g, '').replace(/\s+$/g, '');
}


function wordwrap(str, int_width, str_break, cut) {

	var m = ((arguments.length >= 2) ? arguments[1] : 75);
	var b = ((arguments.length >= 3) ? arguments[2] : "\n");
	var c = ((arguments.length >= 4) ? arguments[3] : false);

	var i, j, l, s, r;

	str += '';

	if (m < 1) {
		return str;
	}
	try {
		for (i = -1, l = (r = str.split(/\r\n|\n|\r/)).length; ++i < l; r[i] += s) {
			for (s = r[i], r[i] = ""; s.length > m; r[i] += s.slice(0, j) + ((s = s.slice(j)).length ? b : "")) {
				j = c == 2 || (j = s.slice(0, m + 1).match(/\S*(\s)?$/))[1] ? m : j.input.length - j[0].length || c == 1 && m || j.input.length + (j = s.slice(m).match(/^\S*/)).input.length;
			}
		}
		return r.join("\n");
	} catch (e) {
		return str;
	}
}


function getAgoString (dateTime) {

	var timespan = new System.TimeSpan();
	var now = monkey.dateTimeNow;
	timespan = now.Subtract(System.DateTime.Parse(dateTime));

	var dayDiff = Math.floor(timespan.TotalDays);
	var secDiff = Math.floor(timespan.TotalSeconds);

	if (dayDiff == 0) {
        // A.
        // Less than one minute ago.
        if (secDiff < 60) {
            return "Just now";
        }
        // B.
        // Less than 2 minutes ago.
        if (secDiff < 120) {
            return "A minute ago";
        }
        // C.
        // Less than one hour ago.
        if (secDiff < 3600) {
            return Math.floor(secDiff / 60) + " min ago";
        }
        // D.
        // Less than 2 hours ago.
        if (secDiff < 7200) {
            return "An hour ago";
        }
        // E.
        // Less than one day ago.
        if (secDiff < 86400) {
            return Math.floor(secDiff / 3600) + " hrs ago";
        }
    }
    else if (dayDiff == 1) {
    	return "A day ago";
    }
    else if (dayDiff > 1) {
    	return dayDiff + " days ago";
    }
}


// Class

function ClipboardItem(text) {

	this.text = "" + text;
	this.menuText = "";
	this.dateCreated = "";
	this.pasted = false;



	this._init = function() {
		this.dateCreated = "" + monkey.dateTimeNow.ToString();
		this.menuText = trim(this.text.replace(/\r\n+|\n+|\t+| {2,}/g, " "));
		this.menuText = wordwrap(this.menuText.substring(0, Math.min(maxMenuItemChars, this.text.length)), startNewLineAt);

		this.menuText += (this.menuText.length >= maxMenuItemChars) ? " ..." : "";
		this.menuText += "\n" + this.text.length + " " + charactersS;
	};
}